## Drawdowns of other assets.
## Have a deemed rate of return on other assets and a deemed rate of saving (calcd from HES). 
## Any difference between actual and expected attributed to drawdowns.


# Prelims -----------------------------------------------------------------

rm(list=ls())
gc()


## custom smoother function
source("./R scripts/Data and parameters/0 Custom functions.R")



# Read in data on rates of return and saving rates ------------------------

other_assets_return_rates <- qread("./Input data/other_asset_returns_a.qs") %>%
  select(-other_assets_return_rate, other_assets_return_rate=other_assets_return_rate_smooth)

other_assets_saving_rates <- qread("./Input data/saving_other_aih.qs") %>%
  select(-age_grp_hes)


# Read in HILDA grouped master data ------------------------------------------------------------

hilda_grouped_master <- qread("./Input data/Intermediate input data/hilda_grouped_master.qs")

wealth_years <- c(6, 10, 14, 18)

hilda_grouped <- hilda_grouped_master %>%
  ## create rough homeowner variable for every year based on housing assets value filled down and upwards - better indicator of INDIVIDUAL homeowner status than yearly hsvalui variable
  mutate(homeowner = case_when(
    housing_assets>0 ~ 1,
    housing_assets==0 ~ 0,
    is.na(housing_assets) ~ NA_real_
  )) %>%
  group_by(xwaveid) %>%
  arrange(xwaveid, wavenumber) %>%
  fill(homeowner, .direction="downup") %>% ## takes a while to run
  ## replace if hsvalui is zero and it is not a wealth year
  mutate(homeowner = case_when(
    hsvalui==0 & !(wavenumber %in% wealth_years) ~ 0,
    ## fill NAs based on hsvalui - at household level rather than indiv level
    is.na(homeowner) & hsvalui>0 & age_grp>="[15,20)" ~ 1,
    is.na(homeowner) ~ 0,
    TRUE ~ homeowner ))

## Non homeowners tend to have fewer other assets than homeowners
mean_other_assets <- hilda_grouped %>% filter(wavenumber %in% wealth_years) %>% group_by(age_grp2, homeowner, wavenumber) %>% summarise(mean = wtd.mean(other_assets, hhwte, na.rm=T))
ggplot(mean_other_assets) + geom_col(aes(x = age_grp2, y=mean, fill=as.factor(homeowner)), position="dodge") + facet_wrap(vars(wavenumber))


# Filter to balanced pooled sample -----------------------------------------------

## for each obs at each wave with wealth data, keep those who are observed in all years up to previous year of wealth data
## this enables change in wealth to be observed, as well as income earned over that time

hilda_grouped_balanced <- lapply(wealth_years, function(x) {
  hilda_grouped %>%
    mutate(total_inc_qtile = as.numeric(total_inc_qtile)) %>% ## for later merging

    ## filter if you appeared in all waves in between wealth years
    filter(wavenumber %in% c((x-4):x)) %>%
    group_by(xwaveid) %>%
    filter(n()==5) %>%
    arrange(xwaveid, wavenumber) %>%

    ## calc TOTAL change in other assets over 4 years, and income earned in past 4 years
    mutate(
           old_other = other_assets[1],
           other_asset_change = (other_assets[5] - other_assets[1]),
           income1 = total_inc[2], ## first year of income to add to savings
           income2 = total_inc[3], ## second year etc
           income3 = total_inc[4],
           income4 = total_inc[5],

           age_grp__1 = age_grp[2], ## age group in first year, for determining what RoR and saving rate to use
           age_grp__2 = age_grp[3], ## in second year etc
           age_grp__3 = age_grp[4],
           age_grp__4 = age_grp[5],
           total_inc_qtile__1 = total_inc_qtile[2], ## inc group in first year for determining what saving rate to use
           total_inc_qtile__2 = total_inc_qtile[3],
           total_inc_qtile__3 = total_inc_qtile[4],
           total_inc_qtile__4 = total_inc_qtile[5],
           homeowner__1 = homeowner[2],
           homeowner__2 = homeowner[3],
           homeowner__3 = homeowner[4],
           homeowner__4 = homeowner[5]
           ) %>%

    filter(wavenumber==x) %>%
    ungroup
}) %>%
  rbindlist %>%
  ## year variable
  mutate(year=wavenumber+2000)


# Calculating drawdowns ---------------------------------------------------

other_asset_drawdowns <- hilda_grouped_balanced %>%
  ## merge in data on assumed saving rates and rates of return for correct age and inc groups each year
  left_join(other_assets_saving_rates %>% setNames(paste0(names(.), "__1"))) %>%
  left_join(other_assets_saving_rates %>% setNames(paste0(names(.), "__2"))) %>%
  left_join(other_assets_saving_rates %>% setNames(paste0(names(.), "__3"))) %>%
  left_join(other_assets_saving_rates %>% setNames(paste0(names(.), "__4"))) %>%
  left_join(other_assets_return_rates %>% setNames(paste0(names(.), "__1"))) %>%
  left_join(other_assets_return_rates %>% setNames(paste0(names(.), "__2"))) %>%
  left_join(other_assets_return_rates %>% setNames(paste0(names(.), "__3"))) %>%
  left_join(other_assets_return_rates %>% setNames(paste0(names(.), "__4"))) %>%
  ## calculate what other assets would have been for each person if we applied our assumed saving rates and rates of return
  mutate(other_assets_expected = old_other*(1+other_assets_return_rate__1)*(1+other_assets_return_rate__2)*(1+other_assets_return_rate__3)*(1+other_assets_return_rate__4)
                                 + income1*saving_rate_other__1*(1+other_assets_return_rate__2)*(1+other_assets_return_rate__3)*(1+other_assets_return_rate__4)
                                 + income2*saving_rate_other__2*(1+other_assets_return_rate__3)*(1+other_assets_return_rate__4)
                                 + income3*saving_rate_other__3*(1+other_assets_return_rate__4)
                                 + income4*saving_rate_other__4,
         ## difference between actual and other
         drawdown = other_assets_expected - other_assets ## positive value means they have less other assets than we expected
                                                         ## negative value means they have more assets than we expected (could be due to gifts/bequests, higher rates of return etc)
           ) %>%  ## this is amount "drawn down" over 4 years
  ## net this all across age groups to get an estimate of overall drawdowns
  group_by(age_grp2, homeowner, wavenumber) %>%
  summarise(drawdown_rate = sum(drawdown*hhwte)/sum(other_assets_expected*hhwte)) %>%
  ## average across waves. Divide by 4 to get a value for 1 year rate
  group_by(age_grp2, homeowner) %>%
  summarise(drawdown_rate = mean(drawdown_rate) / 4) %>% ## negative means they have more assets than we anticipated
  ungroup %>%
  group_by(homeowner) %>%
  arrange(homeowner, age_grp2) %>%
  ## Remove negative vals - edit to 0 (mainly affects very young).  For 85-105 non homeowners, make rate same as 80-85s
  mutate(drawdown_rate = ifelse(drawdown_rate<0, 0, drawdown_rate),
         drawdown_rate = ifelse(age_grp2=="[85,105]" & homeowner==0, lag(drawdown_rate), drawdown_rate))

# ggplot(other_asset_drawdowns) +
#   geom_col(aes(x = age_grp2, y=drawdown_rate, fill=as.factor(homeowner)), position="dodge") 


## smoothing over age
other_asset_drawdowns_smooth <- other_asset_drawdowns %>% 
  group_by(homeowner) %>% 
  arrange(age_grp2, homeowner) %>% 
  ## smoothing
  mutate(across(matches("drawdown_rate$"),  ~custom_smoother(.x) , .names="{.col}_smooth")) 

ggplot(other_asset_drawdowns_smooth) +
  geom_col(aes(x = age_grp2, y=drawdown_rate_smooth, fill=as.factor(homeowner)), position="dodge")
## USED IN APPENDIX:
## These drawdown rates ranged from about 0 to 11 per cent per year. At younger age groups, drawdown rates were 
## relatively high for non-homeowners (who also tended to have fewer other assets) and relatively low for homeowners, 
## and then rates stabilised at older ages. 


summary(other_asset_drawdowns_smooth)

## save
qsave(other_asset_drawdowns_smooth %>% select(age_grp2, homeowner, drawdown_rate_smooth), "./Input data/drawdown_other_ah.qs")

